function iGEM_Model_Final_ANOVA_Wiki()
%% ---------- GLOBAL LOOK & FEEL (Nunito + #ececec) ----------
bg = [236 236 236]/255;  % #ececec
set(groot,'defaultFigureColor',bg);
set(groot,'defaultAxesFontName','Nunito');
set(groot,'defaultTextFontName','Nunito');
set(groot,'defaultLegendFontName','Nunito');

%% ---------- USER INPUT ----------
prompt = {'Total simulation time (hours):','Time step (hours):','Measurement units for counts (e.g., OD600, CFU_per_mL, RFU):'};
dlgtitle = 'Bacterial Growth Model Inputs';
dims = [1 50];
definput = {'16','0.083','Fluorescence intensity'};
answer = inputdlg(prompt,dlgtitle,dims,definput);
if isempty(answer), error('Cancelled.'); end
t_end = str2double(answer{1});
dt    = str2double(answer{2});
units_counts = sanitizeToken(strtrim(answer{3}));
if isempty(units_counts), units_counts = 'units'; end
t_model = 0:dt:t_end;

% --- Custom y-axis label (applies to all plots) ---
default_ylabel = sprintf('%s (a.u.)', units_counts);
answer2 = inputdlg({'Y-axis label for plots:'}, 'Plot Labels', [1 60], {default_ylabel});
if isempty(answer2), error('Cancelled.'); end
y_label = strtrim(answer2{1});
if strlength(y_label) == 0, y_label = default_ylabel; end

%% ---------- RUN TAG ----------
RUN_TAG = datestr(datetime('now'),'yyyymmdd_HHMMSS');

%% ---------- OPTIONS ----------
SAVE_PER_REP_FIGS = true;
PREVIEW_ON_SCREEN = true;
N_PREVIEW = 6;

% --- Fixed Y-limits for ALL time-series plots (log-scale) ---
YLIMS = [5e7, 2e9];   % <<< adjust once here to keep fixed across all figures

% --- Determine save location and file names ---
resultsDir = fullfile(pwd, 'results');
if ~exist(resultsDir,'dir'), mkdir(resultsDir); end

RESULTS_XLSX = fullfile(resultsDir, sprintf('GrowthModel_FittingResults_%s.xlsx', RUN_TAG));
ANOVA_XLSX   = fullfile(resultsDir, sprintf('GrowthModel_ANOVA_%s.xlsx', RUN_TAG));
FIG_FOLDER   = fullfile(resultsDir, ['Replicate_Figures_' RUN_TAG]);
GROUP_FIG_FOLDER = fullfile(resultsDir, ['Group_Figures_' RUN_TAG]);
ANOVA_FIG_FOLDER = fullfile(resultsDir, ['ANOVA_Figures_' RUN_TAG]);

% Fit objective in log-space (good for multiplicative noise)
USE_LOG_FIT = true;

% Percent targets of K:
pct_targets = [0.20 0.30 0.40 0.50 0.60 0.70 0.80 0.90 0.95 0.99];

%% ---------- LOAD FILE + SHEET SELECTION ----------
[file, path] = uigetfile({'*.xlsx;*.csv'}, 'Select Experimental Data File');
if isequal(file,0), error('No file selected.'); end
filepath = fullfile(path,file);

isExcel = endsWith(lower(file),{'.xlsx','.xls','.xlsm','.xlsb'});
selectedSheets = {''};  % default for CSV

if isExcel
    try
        allSheets = sheetnames(filepath);
        [idx, ok] = listdlg('PromptString','Select sheet(s) to process:', ...
                            'SelectionMode','multiple', ...
                            'ListString', allSheets, ...
                            'ListSize',[300 300]);
        if ok && ~isempty(idx)
            selectedSheets = allSheets(idx);
        else
            selectedSheets = allSheets;
        end
    catch
        selectedSheets = {1};
    end
end

%% ---------- OPTIONAL: 96-WELL PLATE LAYOUT ----------
[plateFile, platePath] = uigetfile({'*.xlsx'}, 'Select 96-well Plate Layout (optional)');
plateLayout = [];
HAS_PLATE = ~isequal(plateFile,0);
if HAS_PLATE
    try
        plateLayout = read_plate_layout(fullfile(platePath, plateFile));
        fprintf('Loaded plate layout. Top axis = %s, Left axis = %s\n', ...
            string(plateLayout.axisTop), string(plateLayout.axisLeft));
    catch ME
        warning('iGEM:PlateLayoutReadFailed', ...
                'Failed to read plate layout: %s%sProceeding without mapping.', newline);
        HAS_PLATE = false;
    end
end

%% ---------- OUTPUT FOLDERS ----------
if SAVE_PER_REP_FIGS && ~exist(FIG_FOLDER,'dir'), mkdir(FIG_FOLDER); end
if ~exist(GROUP_FIG_FOLDER,'dir'), mkdir(GROUP_FIG_FOLDER); end
if ~exist(ANOVA_FIG_FOLDER,'dir'), mkdir(ANOVA_FIG_FOLDER); end

%% ---------- LSQCURVEFIT OPTIONS ----------
opts = optimoptions('lsqcurvefit','Display','off','MaxFunctionEvaluations',1e5,'StepTolerance',1e-12,'FunctionTolerance',1e-12);

%% ---------- PREP SUMMARY ----------
model_names = {'Exponential','Logistic','Gompertz'};
best_model_count_global = zeros(1,3);
tPctNames = arrayfun(@(p) sprintf('t_to_%02dpct_of_K_h', round(p*100)), pct_targets, 'UniformOutput', false);
allOutTables = {};

% --- New collectors for group-level results ---
dup_rows_all       = {};   % rows for Condition|Strain duplicate groups (per sheet)
byStrain_rows_all  = {};   % rows pooling all groups with same Strain (per sheet)
byCond_rows_all    = {};   % rows pooling all groups with same Condition (per sheet)

% Collect per-replicate metrics for overall boxplots
RMSE_Exp_all = []; RMSE_Log_all = []; RMSE_Gom_all = [];
AIC_Exp_all  = []; AIC_Log_all  = []; AIC_Gom_all  = [];
R2_Exp_all   = []; R2_Log_all   = []; R2_Gom_all   = [];

%% ---------- PROCESS EACH SHEET ----------
for sh = 1:numel(selectedSheets)

    sheetLabel = selectedSheets{sh};

    % --- Read table; auto-detect header row ---
    if isExcel
        [headerRow, detected] = detect_header_row(filepath, sheetLabel);
        if detected
            fprintf('Detected header row at line %d in sheet %s.\n', headerRow, string(sheetLabel));
            T = readtable(filepath, 'Sheet', sheetLabel, ...
                'VariableNamingRule','preserve', 'TextType','string', ...
                'ReadVariableNames', true, 'HeaderLines', headerRow - 1);
        else
            warning('Could not auto-detect header row in sheet %s. Using first row as header.', string(sheetLabel));
            T = readtable(filepath, 'Sheet', sheetLabel, 'VariableNamingRule','preserve', 'TextType','string');
        end
    else
        T = readtable(filepath, 'VariableNamingRule','preserve', 'TextType','string');
    end
    if size(T,2) < 2
        warning('Sheet %s skipped: expected at least 2 columns.', string(sheetLabel));
        continue;
    end

    % --- Locate 'Time' column dynamically; if not found, prompt user ---
    varNames = T.Properties.VariableNames;
    varNamesLower = lower(varNames);
    timeIdx = find(contains(varNamesLower,'time'), 1, 'first');
    if isempty(timeIdx)
        timeIdx = find(contains(varNamesLower, ["hour","hr","hrs","t (","elapsed","min","minute","minutes"]), 1, 'first');
    end
    if isempty(timeIdx)
        choice = listdlg('PromptString', sprintf('Select the Time column for sheet "%s":', sheetLabel), ...
                         'SelectionMode','single', 'ListString', varNames, 'ListSize',[300 300]);
        if isempty(choice)
            error('No column selected for time in sheet %s.', string(sheetLabel));
        end
        timeIdx = choice;
    end

    % Extract time & replicates (replicates are everything to the right)
    raw_time_col = T.(varNames{timeIdx});
    replicate_names = varNames(timeIdx+1:end);
    num_replicates  = numel(replicate_names);
    if num_replicates < 1
        error('No data columns to the right of the time column in sheet %s.', string(sheetLabel));
    end
    raw_repl = table2cell(T(:, timeIdx+1:end)); % tolerate mixed types

    % Convert replicate block to numeric matrix (NaN for non-numeric)
    N_all = NaN(size(raw_repl,1), size(raw_repl,2));
    for r = 1:size(raw_repl,1)
        for c = 1:size(raw_repl,2)
            v = raw_repl{r,c};
            if isnumeric(v) && isfinite(v)
                N_all(r,c) = double(v);
            elseif isstring(v) || ischar(v)
                vv = str2double(strrep(string(v),',','.'));
                if ~isnan(vv), N_all(r,c) = vv; end
            end
        end
    end

    %% ---------- PARSE TIME ----------
    [t_exp_full, valid_row_mask] = parse_time_column_improved(raw_time_col);
    N_all = N_all(valid_row_mask, :);
    t_exp = t_exp_full(valid_row_mask);

    if isempty(t_exp) || size(N_all,1) ~= numel(t_exp)
        disp('--- Time parser diagnostic (first 10 raw cells) ---');
        nshow = min(10, numel(raw_time_col));
        for k = 1:nshow, disp(raw_time_col(k)); end
        error('No valid time rows, or mismatch between time and data after filtering. Check the time column.');
    end

    EPS = 1e-12;
    N_all(N_all <= 0 | ~isfinite(N_all)) = EPS;

    %% ---------- PREVIEW HEADER ----------
    if PREVIEW_ON_SCREEN
        nprev = min(N_PREVIEW, num_replicates);
        ncols = 2; nrows = ceil(nprev / ncols);
        figure('Name',sprintf('Preview: first %d replicates (%s)', nprev, string(sheetLabel)));
    end

    %% ---------- PER-SHEET SUMMARY ARRAYS ----------
    best_model_count = zeros(1,3);
    rep_col        = strings(num_replicates,1);
    sheet_col      = repmat(string(sheetLabel), num_replicates,1);
    best_model_col = strings(num_replicates,1);
    rmse_exp_col   = nan(num_replicates,1);
    rmse_log_col   = nan(num_replicates,1);
    rmse_gmp_col   = nan(num_replicates,1);
    r_exp_col      = nan(num_replicates,1);
    r_log_col      = nan(num_replicates,1);
    r_gmp_col      = nan(num_replicates,1);
    K_log_col      = nan(num_replicates,1);
    K_gmp_col      = nan(num_replicates,1);
    tL_gmp_col     = nan(num_replicates,1);
    aicc_exp_col   = nan(num_replicates,1);
    aicc_log_col   = nan(num_replicates,1);
    aicc_gmp_col   = nan(num_replicates,1);
    dAICc_exp_col  = nan(num_replicates,1);
    dAICc_log_col  = nan(num_replicates,1);
    dAICc_gmp_col  = nan(num_replicates,1);
    R2_exp_col     = nan(num_replicates,1);
    R2_log_col     = nan(num_replicates,1);
    R2_gmp_col     = nan(num_replicates,1);

    % --- data-derived per-replicate metrics for ANOVA ---
    FoldChange_norm   = nan(num_replicates,1);   % unitless
    MaxDeriv_norm     = nan(num_replicates,1);   % 1/h
    AUC_norm          = nan(num_replicates,1);   % h·unitless

    % --- Plate mapping & grouping metadata ---
    Condition_col     = repmat("Unmapped", num_replicates, 1); % Initial Concentration
    Strain_col        = repmat("Unmapped", num_replicates, 1);
    GroupKey_col      = strings(num_replicates, 1);
    DuplicateIndex    = nan(num_replicates, 1);

    if HAS_PLATE
        for j = 1:num_replicates
            key = char(normKey(replicate_names{j}));
            if isKey(plateLayout.map, key)
                m = plateLayout.map(key);
                Condition_col(j) = m.Condition;  % Initial concentration
                Strain_col(j)    = m.Strain;     % Strain
            end
            GroupKey_col(j) = Condition_col(j) + "|" + Strain_col(j);
        end
        [Gtmp, ~] = findgroups(GroupKey_col);
        for g = 1:max(Gtmp)
            members = find(Gtmp == g);
            for k = 1:numel(members)
                DuplicateIndex(members(k)) = k;
            end
        end
    else
        GroupKey_col = Condition_col + "|" + Strain_col;
        DuplicateIndex(:) = 1;
    end

    tPctCols = nan(num_replicates, numel(pct_targets));

    % --- For group plotting (store per-replicate time+data for this sheet) ---
    rep_tt  = cell(num_replicates,1);
    rep_yy  = cell(num_replicates,1);

    % Valid replicates (skip "Empty")
    validReplicates = ~(strcmpi(Condition_col, "Empty") | strcmpi(Strain_col, "Empty"));

    %% ---------- FIT PER REPLICATE ----------
    for j = 1:num_replicates
        if ~validReplicates(j), continue, end
        rep_col(j) = string(replicate_names{j});

        yy_raw = N_all(:,j);
        mask = ~isnan(yy_raw) & ~isnan(t_exp);
        tt_abs = t_exp(mask);
        yy_all = yy_raw(mask);

        % store for group/ANOVA
        rep_tt{j} = tt_abs;
        rep_yy{j} = yy_all;

        if numel(tt_abs) < 3 || all(~isfinite(yy_all))
            best_model_col(j) = "InsufficientData";
            continue;
        end

        % ----- Use all points (no lag trimming) -----
        tt_fit_abs = tt_abs;
        yy_fit     = yy_all;
        t0 = tt_fit_abs(1);

        % ----- simple normalized metrics (pre-fit) -----
        y0_all = yy_all(1); if ~isfinite(y0_all) || y0_all==0, y0_all = EPS; end
        y_norm_all = yy_all ./ y0_all;
        FoldChange_norm(j) = y_norm_all(end);
        if numel(tt_abs)>=2
            dYdt_all = gradient(y_norm_all, tt_abs); % 1/h
            MaxDeriv_norm(j) = max(dYdt_all);
        end
        AUC_norm(j) = trapz(tt_abs, y_norm_all); % h

        % ----- Initial guesses -----
        N0 = max(yy_fit(1), EPS);

        q_end = t0 + 0.25*(tt_fit_abs(end)-t0);
        early = (tt_fit_abs <= q_end) & (yy_fit > EPS);
        if nnz(early) >= 2
            te = tt_fit_abs(early); ye = yy_fit(early);
            r0 = max(0.05, max( diff(log(ye)) ./ max(diff(te), EPS) ));
        else
            r0 = 0.1;
        end
        if ~isfinite(r0) || r0 <= 0, r0 = 0.1; end

        yMax = max(yy_fit);
        if ~isfinite(yMax) || yMax <= N0
            K0 = max(N0*2, N0+1);
        else
            K0 = max(1.5*yMax, N0 + 1);
        end

        % Gompertz lag parameter (kept internal)
        tL0 = 0;
        tL0_idx = find(diff(yy_fit > 1.05*N0), 1, 'first');
        if ~isempty(tL0_idx)
            tL0 = min(tt_fit_abs(max(1,tL0_idx)) - t0, max(tt_fit_abs) - t0);
        end

        % ----- Model definitions (absolute time; shift inside) -----
        exp_fun  = @(p,t) N0 .* exp(p(1).*(t - t0));
        log_fun  = @(p,t) p(2) ./ (1 + ((p(2)-N0)/max(N0,EPS)) .* exp(-p(1).*(t - t0)));
        gomp_fun = @(p,t) p(2) .* exp(-exp( log(log(max(p(2),N0+EPS)/max(N0,EPS))) - p(1).*((t - t0) - p(3)) ));

        if USE_LOG_FIT
            yy_obj = log(yy_fit + EPS);
            exp_fun_obj  = @(p,t) log(exp_fun(p,t)  + EPS);
            log_fun_obj  = @(p,t) log(log_fun(p,t)  + EPS);
            gomp_fun_obj = @(p,t) log(gomp_fun(p,t) + EPS);
        else
            yy_obj = yy_fit;
            exp_fun_obj  = exp_fun;
            log_fun_obj  = log_fun;
            gomp_fun_obj = gomp_fun;
        end

        % ----- Bounds -----
        lb_exp = 0;                 ub_exp = 10;
        lb_log = [0, max(N0,EPS)];  ub_log = [10, Inf];
        lb_g   = [0, max(N0,EPS), 0];
        ub_g   = [10, Inf, max(tt_fit_abs)-t0];

        % ----- Fits -----
        r_exp=NaN; rmse_exp=Inf; RSS_exp=NaN; AICc_exp=NaN; N_fit_exp_abs=NaN(size(t_model)); R2exp=NaN;
        try
            [p_exp,resnorm] = lsqcurvefit(@(p,t) exp_fun_obj(p,t), r0, tt_fit_abs, yy_obj, lb_exp, ub_exp, opts);
            r_exp = p_exp(1);
            yhat = exp_fun(p_exp,tt_fit_abs);
            rmse_exp = sqrt(mean((yhat - yy_fit).^2));
            N_fit_exp_abs = exp_fun(p_exp, t_model);
            RSS_exp = sum((yhat - yy_fit).^2);
            TSS = sum((yy_fit - mean(yy_fit)).^2);
            if TSS>0, R2exp = 1 - RSS_exp/TSS; end
            AICc_exp = localAICc(resnorm, 1, numel(tt_fit_abs));
        catch, end

        r_log=NaN; K_log=NaN; rmse_log=Inf; RSS_log=NaN; AICc_log=NaN; N_fit_log_abs=NaN(size(t_model)); R2log=NaN;
        try
            [p_log,resnorm] = lsqcurvefit(@(p,t) log_fun_obj(p,t), [r0, K0], tt_fit_abs, yy_obj, lb_log, ub_log, opts);
            r_log = p_log(1); K_log = p_log(2);
            yhat = log_fun(p_log,tt_fit_abs);
            rmse_log = sqrt(mean((yhat - yy_fit).^2));
            N_fit_log_abs = log_fun(p_log, t_model);
            RSS_log = sum((yhat - yy_fit).^2);
            TSS = sum((yy_fit - mean(yy_fit)).^2);
            if TSS>0, R2log = 1 - RSS_log/TSS; end
            AICc_log = localAICc(resnorm, 2, numel(tt_fit_abs));
        catch, end

        r_g=NaN; K_g=NaN; tL_rel=NaN; rmse_g=Inf; RSS_g=NaN; AICc_g=NaN; N_fit_g_abs=NaN(size(t_model)); R2g=NaN;
        try
            [p_g,resnorm] = lsqcurvefit(@(p,t) gomp_fun_obj(p,t), [r0, K0, tL0], tt_fit_abs, yy_obj, lb_g, ub_g, opts);
            r_g = p_g(1); K_g = p_g(2); tL_rel = p_g(3);
            yhat = gomp_fun(p_g,tt_fit_abs);
            rmse_g = sqrt(mean((yhat - yy_fit).^2));
            N_fit_g_abs = gomp_fun(p_g, t_model);
            RSS_g = sum((yhat - yy_fit).^2);
            TSS = sum((yy_fit - mean(yy_fit)).^2);
            if TSS>0, R2g = 1 - RSS_g/TSS; end
            AICc_g = localAICc(resnorm, 3, numel(tt_fit_abs));
        catch, end

        % ----- Select best -----
        aiccs_raw = [AICc_exp, AICc_log, AICc_g];
        valid_aicc = isfinite(aiccs_raw);
        if any(valid_aicc)
            aiccs = aiccs_raw; aiccs(~valid_aicc) = Inf;
            [~, best_idx] = min(aiccs);
            dAICc = aiccs - min(aiccs);
        else
            rmses_raw = [rmse_exp, rmse_log, rmse_g];
            valid_rmse = isfinite(rmses_raw);
            if any(valid_rmse)
                rmses = rmses_raw; rmses(~valid_rmse) = Inf;
                [~, best_idx] = min(rmses); dAICc = [NaN NaN NaN];
            else
                best_idx = NaN; dAICc = [NaN NaN NaN];
            end
        end

        if ~isfinite(best_idx)
            best_model_col(j) = "InsufficientData";
        else
            best_model_col(j) = string(model_names{best_idx});
            best_model_count(best_idx) = best_model_count(best_idx)+1;
        end

        % ----- Record metrics -----
        rmse_exp_col(j)   = rmse_exp;
        rmse_log_col(j)   = rmse_log;
        rmse_gmp_col(j)   = rmse_g;

        r_exp_col(j)      = r_exp;
        r_log_col(j)      = r_log;
        r_gmp_col(j)      = r_g;
        K_log_col(j)      = K_log;
        K_gmp_col(j)      = K_g;
        tL_gmp_col(j)     = tL_rel;

        aicc_exp_col(j)   = AICc_exp;
        aicc_log_col(j)   = AICc_log;
        aicc_gmp_col(j)   = AICc_g;
        dAICc_exp_col(j)  = dAICc(1);
        dAICc_log_col(j)  = dAICc(2);
        dAICc_gmp_col(j)  = dAICc(3);

        R2_exp_col(j)     = R2exp;
        R2_log_col(j)     = R2log;
        R2_gmp_col(j)     = R2g;

        % also append to global arrays (for summary boxplots later)
        if isfinite(rmse_exp), RMSE_Exp_all(end+1) = rmse_exp; end %#ok<AGROW>
        if isfinite(rmse_log), RMSE_Log_all(end+1) = rmse_log; end %#ok<AGROW>
        if isfinite(rmse_g ), RMSE_Gom_all(end+1) = rmse_g;   end %#ok<AGROW>
        if isfinite(AICc_exp), AIC_Exp_all(end+1) = AICc_exp; end %#ok<AGROW>
        if isfinite(AICc_log), AIC_Log_all(end+1) = AICc_log; end %#ok<AGROW>
        if isfinite(AICc_g  ), AIC_Gom_all(end+1) = AICc_g;   end %#ok<AGROW>
        if isfinite(R2exp), R2_Exp_all(end+1) = R2exp; end %#ok<AGROW>
        if isfinite(R2log), R2_Log_all(end+1) = R2log; end %#ok<AGROW>
        if isfinite(R2g  ), R2_Gom_all(end+1) = R2g;   end %#ok<AGROW>

        % ----- Time-to-%K (for logistic/gompertz) -----
        t_to_pct = nan(1, numel(pct_targets));
        switch best_idx
            case 2
                for k = 1:numel(pct_targets)
                    p = pct_targets(k);
                    t_rel = timeToPctK_Logistic(p, r_log, K_log, N0, EPS);
                    if isfinite(t_rel), t_to_pct(k) = t0 + t_rel; end
                end
            case 3
                for k = 1:numel(pct_targets)
                    p = pct_targets(k);
                    t_rel = timeToPctK_Gompertz(p, r_g, K_g, N0, max(0,tL_rel), EPS);
                    if isfinite(t_rel), t_to_pct(k) = t0 + t_rel; end
                end
        end
        tPctCols(j, :) = t_to_pct;

        % ----- Save figure per replicate -----
        if SAVE_PER_REP_FIGS
            fSafe = safeFilename(sprintf('%s__%s', string(sheetLabel), replicate_names{j}));
            if isfinite(best_idx), best_name = model_names{best_idx}; else, best_name = 'NA'; end
            fBest = sprintf('%s__best-%s__%s.png', char(fSafe), best_name, RUN_TAG);

            fig = figure('Visible','off'); hold on;
            yy_plot_mask = yy_all > EPS;
            plot(tt_abs(yy_plot_mask), yy_all(yy_plot_mask), 'o','DisplayName','Experimental');
            if all(isfinite(N_fit_exp_abs)),  plot(t_model, N_fit_exp_abs,  '-','DisplayName','Exponential'); end
            if all(isfinite(N_fit_log_abs)),  plot(t_model, N_fit_log_abs,  '-','DisplayName','Logistic');    end
            if all(isfinite(N_fit_g_abs)),    plot(t_model, N_fit_g_abs,    '-','DisplayName','Gompertz');   end

            if isfinite(tL_gmp_col(j)) && isfinite(K_gmp_col(j))
                yline(K_gmp_col(j), '--', sprintf('K = %.3g %s', K_gmp_col(j), units_counts), ...
                    'LabelHorizontalAlignment','left','LabelVerticalAlignment','top','DisplayName','K');
            elseif isfinite(K_log_col(j))
                yline(K_log_col(j), '--', sprintf('K = %.3g %s', K_log_col(j), units_counts), ...
                    'LabelHorizontalAlignment','left','LabelVerticalAlignment','top','DisplayName','K');
            end

            set(gca, 'fontsize',16);
            xlabel('Time (hours)'); ylabel(y_label);
            title(sprintf('Sheet %s | %s (dup %s) | Rep %s | Best: %s', ...
                char(string(sheetLabel)), char(string(Condition_col(j) + " | " + Strain_col(j))), ...
                num2str(DuplicateIndex(j)), char(string(replicate_names{j})), best_name));
            bump_title(gca, 0.02);
            set(gca,'YScale','log'); grid on; legend('Location','best');
            ylim(YLIMS);
            try
                saveas(fig, fullfile(FIG_FOLDER, fBest));
            catch ME
                warning('Saving figure failed (%s): %s', fBest);
            end
            close(fig);
        end

        % ----- On-screen preview -----
        if PREVIEW_ON_SCREEN && exist('nprev','var') && j <= nprev
            if isfinite(best_idx), best_name = model_names{best_idx}; else, best_name = 'NA'; end
            subplot(nrows, ncols, j); hold on;
            yy_plot_mask = yy_all > EPS;
            plot(tt_abs(yy_plot_mask), yy_all(yy_plot_mask), 'o');
            if all(isfinite(N_fit_exp_abs)),  plot(t_model, N_fit_exp_abs,  '-'); end
            if all(isfinite(N_fit_log_abs)),  plot(t_model, N_fit_log_abs,  '-'); end
            if all(isfinite(N_fit_g_abs)),    plot(t_model, N_fit_g_abs,    '-'); end
            if isfinite(K_log_col(j))
                yline(K_log_col(j), '--', sprintf('K = %.3g %s', K_log_col(j), units_counts), ...
                    'LabelHorizontalAlignment','left','LabelVerticalAlignment','top');
            end
            xlabel('Time (hours)'); ylabel(y_label);
            title(sprintf('%s | %s (dup %s) | Best: %s', ...
                char(string(replicate_names{j})), char(string(Condition_col(j) + " | " + Strain_col(j))), ...
                num2str(DuplicateIndex(j)), best_name));
            bump_title(gca, 0.02);
            set(gca,'YScale','log'); grid on;
            ylim(YLIMS);
        end

    end % replicate loop

    %% ---------- GROUP (DUPLICATE) PLOTS + GROUP MODEL FIT ----------
    [G, groupKeys] = findgroups(GroupKey_col);

    % --- Store mean traces for reuse in composite plots ---
    group_mean_t = containers.Map('KeyType','char','ValueType','any'); % key = "Condition|Strain"
    group_mean_y = containers.Map('KeyType','char','ValueType','any');
    group_cond   = containers.Map('KeyType','char','ValueType','any'); % Condition label
    group_strain = containers.Map('KeyType','char','ValueType','any'); % Strain label

for g = 1:max(G)
    members = find(G == g);
    if numel(members) < 2, continue; end

    % --- derive Condition / Strain from the group key (SAFE) ---
    groupLabel = string(groupKeys(g));   % "Condition|Strain"
    tokens = split(groupLabel, "|");

    % ... (now proceed to build t_grp / y_grp and do fits) ...


            groupLabel = string(groupKeys(g)); % "Condition|Strain"
    cond_strain = split(groupLabel,"|");
    if numel(cond_strain) < 2
        condLab = groupLabel; 
        strainLab = "";
    else
        condLab   = strtrim(cond_strain(1));
        strainLab = strtrim(cond_strain(2));
    end

        % --- Skip Empty groups early ---
        if strcmpi(condLab, "Empty") || strcmpi(strainLab, "Empty") || ...
           strlength(condLab) == 0 || strlength(strainLab) == 0
            continue;   % skip this group entirely
        end

        % ---- Build a group trace ----
        sameTime = true;
        t_ref = rep_tt{members(1)};
        for k = 2:numel(members)
            tk = rep_tt{members(k)};
            if numel(tk) ~= numel(t_ref) || any(abs(tk - t_ref) > 1e-9)
                sameTime = false; break;
            end
        end

        if sameTime
            Ystack = zeros(numel(t_ref), numel(members));
            for k = 1:numel(members)
                Ystack(:,k) = rep_yy{members(k)};
            end
            t_grp = t_ref(:);
            y_grp = mean(Ystack, 2, 'omitnan');
        else
            t_grp = []; y_grp = [];
            for k = 1:numel(members)
                t_grp = [t_grp; rep_tt{members(k)}(:)]; %#ok<AGROW>
                y_grp = [y_grp; rep_yy{members(k)}(:)]; %#ok<AGROW>
            end
            [t_grp, order] = sort(t_grp);
            y_grp = y_grp(order);
        end

        % ---- Save a table row for this duplicate group (Condition|Strain) ----
        % keyA = Condition, keyB = Strain
        [row_dup, ~] = fit_and_tabulate_group(t_grp, y_grp, sheetLabel, condLab, strainLab, ...
            units_counts, pct_targets, tPctNames, t_model, USE_LOG_FIT, opts, EPS);
        
        dup_rows_all{end+1} = row_dup; %#ok<AGROW>

        maskG = isfinite(t_grp) & isfinite(y_grp);
        t_grp = t_grp(maskG); y_grp = y_grp(maskG);
        if numel(t_grp) < 3, continue; end

        % Save mean trace for composite plots
        gkey = char(string(groupLabel));
        group_mean_t(gkey) = t_grp;
        group_mean_y(gkey) = y_grp;
        group_cond(gkey)   = string(condLab);
        group_strain(gkey) = string(strainLab);

        % ---- Fit models to the group trace ----
        tt_fit_g = t_grp;
        yy_fit_g = y_grp;
        if numel(tt_fit_g) < 3, continue; end
        t0g = tt_fit_g(1);
        N0g = max(yy_fit_g(1), EPS);

        q_end = t0g + 0.25*(tt_fit_g(end)-t0g);
        early = (tt_fit_g <= q_end) & (yy_fit_g > EPS);
        if nnz(early) >= 2
            te = tt_fit_g(early); ye = yy_fit_g(early);
            r0g = max(0.05, max(diff(log(ye))./max(diff(te),EPS)));
        else
            r0g = 0.1;
        end
        if ~isfinite(r0g) || r0g <= 0, r0g = 0.1; end
        yMaxg = max(yy_fit_g);
        K0g = (isfinite(yMaxg) && yMaxg > N0g) * max(1.5*yMaxg, N0g + 1) + (~(isfinite(yMaxg) && yMaxg > N0g)) * max(N0g*2, N0g+1);
        tL0g = 0;
        tL0_idx = find(diff(yy_fit_g > 1.05*N0g), 1, 'first');
        if ~isempty(tL0_idx)
            tL0g = min(tt_fit_g(max(1,tL0_idx)) - t0g, max(tt_fit_g) - t0g);
        end

        exp_fun_g  = @(p,t) N0g .* exp(p(1).*(t - t0g));
        log_fun_g  = @(p,t) p(2) ./ (1 + ((p(2)-N0g)/max(N0g,EPS)) .* exp(-p(1).*(t - t0g)));
        gomp_fun_g = @(p,t) p(2) .* exp(-exp( log(log(max(p(2),N0g+EPS)/max(N0g,EPS))) - p(1).*((t - t0g) - p(3)) ));

        if USE_LOG_FIT
            yy_obj_g = log(yy_fit_g + EPS);
            exp_fun_obj_g  = @(p,t) log(exp_fun_g(p,t)  + EPS);
            log_fun_obj_g  = @(p,t) log(log_fun_g(p,t)  + EPS);
            gomp_fun_obj_g = @(p,t) log(gomp_fun_g(p,t) + EPS);
        else
            yy_obj_g = yy_fit_g;
            exp_fun_obj_g  = exp_fun_g;
            log_fun_obj_g  = log_fun_g;
            gomp_fun_obj_g = gomp_fun_g;
        end

        lb_exp_g = 0;                ub_exp_g = 10;
        lb_log_g = [0, max(N0g,EPS)]; ub_log_g = [10, Inf];
        lb_g_g   = [0, max(N0g,EPS), 0];
        ub_g_g   = [10, Inf, max(tt_fit_g)-t0g];

        rE=NaN; rmE=Inf; aE=NaN; Nexp=NaN(size(t_model));
        rL=NaN; KL=NaN; rmL=Inf; aL=NaN; Nlogi=NaN(size(t_model));
        rG=NaN; KG=NaN; tLG=NaN; rmG=Inf; aG=NaN; Ngomp=NaN(size(t_model));

        try
            [p,res] = lsqcurvefit(@(p,t) exp_fun_obj_g(p,t), r0g, tt_fit_g, yy_obj_g, lb_exp_g, ub_exp_g, opts);
            rE = p(1); rmE = sqrt(mean((exp_fun_g(p,tt_fit_g)-yy_fit_g).^2)); aE = localAICc(res,1,numel(tt_fit_g)); Nexp = exp_fun_g(p,t_model);
        catch, end
        try
            [p,res] = lsqcurvefit(@(p,t) log_fun_obj_g(p,t), [r0g,K0g], tt_fit_g, yy_obj_g, lb_log_g, ub_log_g, opts);
            rL = p(1); KL = p(2); rmL = sqrt(mean((log_fun_g(p,tt_fit_g)-yy_fit_g).^2)); aL = localAICc(res,2,numel(tt_fit_g)); Nlogi = log_fun_g(p,t_model);
        catch, end
        try
            [p,res] = lsqcurvefit(@(p,t) gomp_fun_obj_g(p,t), [r0g,K0g,tL0g], tt_fit_g, yy_obj_g, lb_g_g, ub_g_g, opts);
            rG = p(1); KG = p(2); tLG = p(3); rmG = sqrt(mean((gomp_fun_g(p,tt_fit_g)-yy_fit_g).^2)); aG = localAICc(res,3,numel(tt_fit_g)); Ngomp = gomp_fun_g(p,t_model);
        catch, end

        A = [aE aL aG]; valid = isfinite(A);
        if any(valid)
            A(~valid) = Inf; [~, bestG] = min(A);
        else
            R = [rmE rmL rmG]; validR = isfinite(R); R(~validR)=Inf; [~, bestG] = min(R);
        end

        % ---- Plot group graph with fits ----
        fig = figure('Name', sprintf('Sheet %s | %s | Duplicates', string(sheetLabel), string(groupLabel)), 'Visible','off'); hold on;

        % plot each replicate
        for k = 1:numel(members)
            jj = members(k);
            plot(rep_tt{jj}, rep_yy{jj}, 'o-', 'DisplayName', sprintf('%s (dup %d)', char(rep_col(jj)), k));
        end

        % overlay group trace & models
        plot(t_grp, y_grp, '.', 'DisplayName', 'Group trace');
        if all(isfinite(Nexp)),  plot(t_model, Nexp,  '-','DisplayName','Exp (group fit)'); end
        if all(isfinite(Nlogi)), plot(t_model, Nlogi, '-','DisplayName','Logistic (group fit)'); end
        if all(isfinite(Ngomp)), plot(t_model, Ngomp, '-','DisplayName','Gompertz (group fit)'); end

        xlabel('Time (hours)'); ylabel(y_label);
        title(sprintf('Sheet %s | %s | %s | Duplicates (n=%d) + Group Fit', ...
            string(sheetLabel), string(condLab), string(strainLab), numel(members)));
        bump_title(gca, 0.02);
        set(gca,'YScale','log'); grid on; legend('Location','best');
        ylim(YLIMS);
        fSafe = safeFilename(sprintf('Sheet_%s__Group_%s', string(sheetLabel), string(groupLabel)));
        try
            saveas(fig, fullfile(GROUP_FIG_FOLDER, sprintf('%s__%s.png', char(fSafe), RUN_TAG)));
        catch ME
            warning('Saving group figure failed: %s');
        end
        close(fig);

    end % group loop

    %% ---------- COMPOSITE FIGURES USING MEAN OF DUPLICATES ----------
    allKeys = group_mean_t.keys;
    if ~isempty(allKeys)
        % Unique lists from stored groups
        strains_list    = unique(string(values(group_strain, allKeys)));
        conditions_list = unique(string(values(group_cond,   allKeys)));

        % --- SAME STRAIN: curves per Condition (legend = Condition) ---
        for sidx = 1:numel(strains_list)
            sname = strains_list(sidx);
            if any(strcmpi(sname, ["Unmapped","Empty",""])), continue; end

            keys_this_strain = {};
            cond_labels = string.empty(0,1);
            for k = 1:numel(allKeys)
                kk = allKeys{k};
                if group_strain(kk) == sname
                    keys_this_strain{end+1} = kk; %#ok<AGROW>
                    cond_labels(end+1,1)    = group_cond(kk); %#ok<AGROW>
                end
            end
            if isempty(keys_this_strain), continue; end

            f = figure('Visible','off'); hold on;
            for i = 1:numel(keys_this_strain)
                kk  = keys_this_strain{i};
                tt  = group_mean_t(kk);
                yy  = group_mean_y(kk);
                if isempty(tt) || isempty(yy), continue; end
                plot(tt, yy, 'o-', 'DisplayName', char(cond_labels(i)));   % legend = Condition
            end
            set(gca,'YScale','log'); grid on;
            xlabel('Time (hours)'); ylabel(y_label);
            title(sprintf('Sheet %s | Strain: %s | Mean duplicate curves by Condition', ...
                  char(string(sheetLabel)), char(sname)));
            bump_title(gca, 0.02);
            legend('Location','best');
            ylim(YLIMS);
            fSafe = safeFilename(sprintf('Sheet_%s__ByStrain_Mean_%s', ...
                   char(string(sheetLabel)), char(sname)));
            try
                saveas(f, fullfile(GROUP_FIG_FOLDER, sprintf('%s__%s.png', char(fSafe), RUN_TAG)));
            catch ME
                warning('Saving composite (strain) failed: %s');
            end
            close(f);
            % ---- Fit pooled data for this Strain across all Conditions ----
            t_pool = []; y_pool = [];
            for i = 1:numel(keys_this_strain)
                kk = keys_this_strain{i};
                tt = group_mean_t(kk);   % <-- temp
                yy = group_mean_y(kk);   % <-- temp
                t_pool = [t_pool; tt(:)];
                y_pool = [y_pool; yy(:)];
            end
            [t_pool, ord] = sort(t_pool);
            y_pool = y_pool(ord);
            [row_bs, ~] = fit_and_tabulate_group(t_pool, y_pool, sheetLabel, sname, "(All Conditions)", ...
                units_counts, pct_targets, tPctNames, t_model, USE_LOG_FIT, opts, EPS);
            byStrain_rows_all{end+1} = row_bs; %#ok<AGROW>
        end

        % --- SAME CONDITION: curves per Strain (legend = Strain) ---
        for cidx = 1:numel(conditions_list)
            cname = conditions_list(cidx);
            if any(strcmpi(cname, ["Unmapped","Empty",""])), continue; end

            keys_this_cond = {};
            strain_labels = string.empty(0,1);
            for k = 1:numel(allKeys)
                kk = allKeys{k};
                if group_cond(kk) == cname
                    keys_this_cond{end+1} = kk; %#ok<AGROW>
                    strain_labels(end+1,1) = group_strain(kk); %#ok<AGROW>
                end
            end
            if isempty(keys_this_cond), continue; end

            f = figure('Visible','off'); hold on;
            for i = 1:numel(keys_this_cond)
                kk  = keys_this_cond{i};
                tt  = group_mean_t(kk);
                yy  = group_mean_y(kk);
                if isempty(tt) || isempty(yy), continue; end
                plot(tt, yy, 'o-', 'DisplayName', char(strain_labels(i))); % legend = Strain
            end
            set(gca,'YScale','log'); grid on;
            xlabel('Time (hours)'); ylabel(y_label);
            title(sprintf('Sheet %s | Condition: %s | Mean duplicate curves by Strain', ...
                  char(string(sheetLabel)), char(cname)));
            bump_title(gca, 0.02);
            legend('Location','best');
            ylim(YLIMS);
            fSafe = safeFilename(sprintf('Sheet_%s__ByCondition_Mean_%s', ...
                   char(string(sheetLabel)), char(cname)));
            try
                saveas(f, fullfile(GROUP_FIG_FOLDER, sprintf('%s__%s.png', char(fSafe), RUN_TAG)));
            catch ME
                warning('Saving composite (condition) failed: %s');
            end
            close(f);
            % ---- Fit pooled data for this Condition across all Strains ----
            t_pool = []; y_pool = [];
            for i = 1:numel(keys_this_cond)
                kk = keys_this_cond{i};
                tt = group_mean_t(kk);   % <-- temp
                yy = group_mean_y(kk);   % <-- temp
                t_pool = [t_pool; tt(:)];
                y_pool = [y_pool; yy(:)];
            end
            [t_pool, ord] = sort(t_pool);
            y_pool = y_pool(ord);
            [row_bc, ~] = fit_and_tabulate_group(t_pool, y_pool, sheetLabel, cname, "(All Strains)", ...
                units_counts, pct_targets, tPctNames, t_model, USE_LOG_FIT, opts, EPS);
            byCond_rows_all{end+1} = row_bc; %#ok<AGROW>
        end
    end

    %% ---------- BUILD PER-SHEET TABLE ----------
    Out = table( ...
        sheet_col, rep_col, ...
        Condition_col, Strain_col, GroupKey_col, DuplicateIndex, ...
        best_model_col, ...
        rmse_exp_col, rmse_log_col, rmse_gmp_col, ...
        r_exp_col, r_log_col, r_gmp_col, ...
        K_log_col, K_gmp_col, tL_gmp_col, ...
        aicc_exp_col, aicc_log_col, aicc_gmp_col, ...
        dAICc_exp_col, dAICc_log_col, dAICc_gmp_col, ...
        R2_exp_col, R2_log_col, R2_gmp_col, ...
        FoldChange_norm, MaxDeriv_norm, AUC_norm, ...
        'VariableNames', { ...
            'Sheet','Replicate', ...
            'Condition','Strain','GroupKey','DuplicateIndex', ...
            'BestModel', ...
            ['RMSE_Exp_' units_counts], ['RMSE_Log_' units_counts], ['RMSE_Gomp_' units_counts], ...
            'r_Exp_1_per_h','r_Log_1_per_h','r_Gomp_1_per_h', ...
            ['K_Log_' units_counts], ['K_Gomp_' units_counts], 'tL_Gomp_rel_h', ...
            'AICc_Exp','AICc_Log','AICc_Gomp', ...
            'dAICc_Exp','dAICc_Log','dAICc_Gomp', ...
            'R2_Exp','R2_Log','R2_Gomp', ...
            'FoldChange_norm','MaxDeriv_norm_1_per_h','AUC_norm_h' ...
        } ...
    );

    for k = 1:numel(pct_targets)
        Out.(tPctNames{k}) = tPctCols(:,k);
    end
    Out.Criteria = repmat("AICc log-space; absolute time (t−t0). Plate-mapped groups + duplicates. Per-rep K line. Duplicate/composite graphs included. Summary boxplots and ANOVA exported.", height(Out), 1);

    allOutTables{end+1} = Out;
    best_model_count_global = best_model_count_global + best_model_count;

end % sheets loop

%% ---------- WRITE EXCEL SUMMARY ----------
if isempty(allOutTables)
    error('No valid sheets processed.');
end
OutAll = vertcat(allOutTables{:});
writetable(OutAll, RESULTS_XLSX);
fprintf('Saved summary to %s\n', RESULTS_XLSX);

%% ---------- WRITE GROUP-LEVEL FITS TO EXCEL (new sheets) ----------
% Column names for grouped tables
colNames = { ...
    'Sheet','KeyA','KeyB','N_points','BestModel', ...
    ['RMSE_Exp_' units_counts], ['RMSE_Log_' units_counts], ['RMSE_Gomp_' units_counts], ...
    'r_Exp_1_per_h','r_Log_1_per_h','r_Gomp_1_per_h', ...
    ['K_Log_' units_counts], ['K_Gomp_' units_counts], 'tL_Gomp_rel_h', ...
    'AICc_Exp','AICc_Log','AICc_Gomp', ...
    'dAICc_Exp','dAICc_Log','dAICc_Gomp', ...
    'R2_Exp','R2_Log','R2_Gomp' ...
};
for k = 1:numel(tPctNames), colNames{end+1} = tPctNames{k}; end
expectedCols = numel(colNames);

% Helper: make each row 1×expectedCols (pad/truncate), then stack to m×n
function C = rows_to_matrix(rows, expectedCols)
    if isempty(rows), C = {}; return; end
    rows = rows(:);
    for ii = 1:numel(rows)
        r = rows{ii};
        if isempty(r), r = {}; end
        if ~isrow(r), r = r(:).'; end
        n = numel(r);
        if n < expectedCols
            r = [r, num2cell(nan(1, expectedCols - n))];
        elseif n > expectedCols
            r = r(1:expectedCols);
        end
        rows{ii} = r;
    end
    C = vertcat(rows{:});
end

% ---- Duplicates (Condition|Strain) ----
if ~isempty(dup_rows_all)
    Cdup = rows_to_matrix(dup_rows_all, expectedCols);
    if ~isempty(Cdup)
        T_dup = cell2table(Cdup, 'VariableNames', colNames);
        T_dup.Properties.VariableNames{'KeyA'} = 'Condition';
        T_dup.Properties.VariableNames{'KeyB'} = 'Strain';
        writetable(T_dup, RESULTS_XLSX, 'Sheet', 'Group_Duplicates');
    end
end

% ---- By-Strain (pool over all Conditions) ----
if ~isempty(byStrain_rows_all)
    Cbs = rows_to_matrix(byStrain_rows_all, expectedCols);
    if ~isempty(Cbs)
        T_bs = cell2table(Cbs, 'VariableNames', colNames);
        T_bs.Properties.VariableNames{'KeyA'} = 'Strain';
        T_bs.Properties.VariableNames{'KeyB'} = 'GroupNote';
        writetable(T_bs, RESULTS_XLSX, 'Sheet', 'Group_ByStrain');
    end
end

% ---- By-Condition (pool over all Strains) ----
if ~isempty(byCond_rows_all)
    Cbc = rows_to_matrix(byCond_rows_all, expectedCols);
    if ~isempty(Cbc)
        T_bc = cell2table(Cbc, 'VariableNames', colNames);
        T_bc.Properties.VariableNames{'KeyA'} = 'Condition';
        T_bc.Properties.VariableNames{'KeyB'} = 'GroupNote';
        writetable(T_bc, RESULTS_XLSX, 'Sheet', 'Group_ByCondition');
    end
end

%% ---------- BAR CHART ----------
fig = figure('Name','Best model count (all sheets)','Visible','off');
bar(best_model_count_global);
set(gca,'XTickLabel',model_names);
set(gca, 'fontsize',16);
ylabel('Number of Replicates'); title('Best-Fit Model Count per Replicate (by AICc)');
bump_title(gca, 0.02);
grid on;
try, saveas(fig, fullfile(ANOVA_FIG_FOLDER, sprintf('BestModel_Counts__%s.png', RUN_TAG))); end
close(fig);

%% ---------- SUMMARY BOX & WHISKER: RMSE, AICc, R² (all replicates) ----------
make_summary_boxplots( ...
    RMSE_Exp_all, RMSE_Log_all, RMSE_Gom_all, ...
    AIC_Exp_all,  AIC_Log_all,  AIC_Gom_all,  ...
    R2_Exp_all,   R2_Log_all,   R2_Gom_all,   ...
    ANOVA_FIG_FOLDER, RUN_TAG);

function [row, yhatStruct] = fit_and_tabulate_group(tt, yy, sheetLabel, keyA, keyB, units_counts, ...
                                                     pct_targets, tPctNames, t_model, USE_LOG_FIT, opts, EPS)
yhatStruct = struct('exp',NaN(size(t_model)),'log',NaN(size(t_model)),'gom',NaN(size(t_model)));

nPct = numel(pct_targets);
baseRow = { string(sheetLabel), string(keyA), string(keyB), ...
            NaN, "NA", ...
            NaN, NaN, NaN, ...   % RMSEs
            NaN, NaN, NaN, ...   % r's
            NaN, NaN, NaN, ...   % K_log, K_gomp, tL_gomp
            NaN, NaN, NaN, ...   % AICc's
            NaN, NaN, NaN, ...   % dAICc's
            NaN, NaN, NaN };     % R2's
defaultPct = num2cell(nan(1, nPct));

% Early exits must still return full-length row
if numel(tt) < 3 || all(~isfinite(yy))
    row = [baseRow, defaultPct];
    return;
end

tt = tt(:); yy = yy(:);
mask = isfinite(tt) & isfinite(yy) & (yy > 0);
tt = tt(mask); yy = yy(mask);
if numel(tt) < 3
    row = [baseRow, defaultPct];
    return;
end

t0  = tt(1);
N0  = max(yy(1), EPS);

% crude r0 from earliest 25%
q_end = t0 + 0.25*(tt(end)-t0);
early = (tt <= q_end) & (yy > EPS);
if nnz(early) >= 2
    te = tt(early); ye = yy(early);
    r0 = max(0.05, max(diff(log(ye))./max(diff(te),EPS)));
else
    r0 = 0.1;
end
if ~isfinite(r0) || r0 <= 0, r0 = 0.1; end

yMax = max(yy);
if ~isfinite(yMax) || yMax <= N0, K0 = max(N0*2, N0+1); else, K0 = max(1.5*yMax, N0+1); end
tL0 = 0; tL_idx = find(diff(yy > 1.05*N0), 1, 'first');
if ~isempty(tL_idx)
    tL0 = min(tt(max(1,tL_idx)) - t0, max(tt) - t0);
end

% models
exp_fun  = @(p,t) N0 .* exp(p(1).*(t - t0));
log_fun  = @(p,t) p(2) ./ (1 + ((p(2)-N0)/max(N0,EPS)) .* exp(-p(1).*(t - t0)));
gomp_fun = @(p,t) p(2) .* exp(-exp( log(log(max(p(2),N0+EPS)/max(N0,EPS))) - p(1).*((t - t0) - p(3)) ));

if USE_LOG_FIT
    yy_obj        = log(yy + EPS);
    exp_fun_obj   = @(p,t) log(exp_fun(p,t)  + EPS);
    log_fun_obj   = @(p,t) log(log_fun(p,t)  + EPS);
    gomp_fun_obj  = @(p,t) log(gomp_fun(p,t) + EPS);
else
    yy_obj        = yy;
    exp_fun_obj   = exp_fun;
    log_fun_obj   = log_fun;
    gomp_fun_obj  = gomp_fun;
end

% bounds
lb_exp = 0;               ub_exp = 10;
lb_log = [0, max(N0,EPS)]; ub_log = [10, Inf];
lb_g   = [0, max(N0,EPS), 0];
ub_g   = [10, Inf, max(tt)-t0];

% fit all three
rE=NaN; rmE=Inf; aE=NaN; R2E=NaN;
rL=NaN; KL=NaN; rmL=Inf; aL=NaN; R2L=NaN;
rG=NaN; KG=NaN; tLG=NaN; rmG=Inf; aG=NaN; R2G=NaN;

try
    [p,res] = lsqcurvefit(@(p,t) exp_fun_obj(p,t), r0, tt, yy_obj, lb_exp, ub_exp, opts);
    rE = p(1);
    yhat = exp_fun(p,tt);
    rmE  = sqrt(mean((yhat - yy).^2));
    RSS  = sum((yhat - yy).^2); TSS = sum((yy - mean(yy)).^2); if TSS>0, R2E = 1 - RSS/TSS; end
    aE   = localAICc(res, 1, numel(tt));
    yhatStruct.exp = exp_fun(p, t_model);
catch, end

try
    [p,res] = lsqcurvefit(@(p,t) log_fun_obj(p,t), [r0,K0], tt, yy_obj, lb_log, ub_log, opts);
    rL = p(1); KL = p(2);
    yhat = log_fun(p,tt);
    rmL  = sqrt(mean((yhat - yy).^2));
    RSS  = sum((yhat - yy).^2); TSS = sum((yy - mean(yy)).^2); if TSS>0, R2L = 1 - RSS/TSS; end
    aL   = localAICc(res, 2, numel(tt));
    yhatStruct.log = log_fun(p, t_model);
catch, end

try
    [p,res] = lsqcurvefit(@(p,t) gomp_fun_obj(p,t), [r0,K0,tL0], tt, yy_obj, lb_g, ub_g, opts);
    rG = p(1); KG = p(2); tLG = p(3);
    yhat = gomp_fun(p,tt);
    rmG  = sqrt(mean((yhat - yy).^2));
    RSS  = sum((yhat - yy).^2); TSS = sum((yy - mean(yy)).^2); if TSS>0, R2G = 1 - RSS/TSS; end
    aG   = localAICc(res, 3, numel(tt));
    yhatStruct.gom = gomp_fun(p, t_model);
catch, end

A   = [aE aL aG];
okA = isfinite(A);
if any(okA)
    A(~okA)=Inf; [~,best] = min(A); dA = A - min(A);
else
    RMS  = [rmE rmL rmG];
    okR  = isfinite(RMS);
    if any(okR), RMS(~okR)=Inf; [~,best] = min(RMS); dA = [NaN NaN NaN];
    else, best = NaN; dA = [NaN NaN NaN];
    end
end

names = ["Exponential","Logistic","Gompertz"];
if isnumeric(best) && isfinite(best) && best>=1 && best<=numel(names)
    bestName = names(best);      % string scalar
else
    bestName = "NA";
end

% time-to-%K
t_to_pct = nan(1, numel(pct_targets));
switch best
    case 2
        for k=1:numel(pct_targets)
            t_rel = timeToPctK_Logistic(pct_targets(k), rL, KL, N0, EPS);
            if isfinite(t_rel), t_to_pct(k) = t0 + t_rel; end
        end
    case 3
        for k=1:numel(pct_targets)
            t_rel = timeToPctK_Gompertz(pct_targets(k), rG, KG, N0, max(0,tLG), EPS);
            if isfinite(t_rel), t_to_pct(k) = t0 + t_rel; end
        end
end

% assemble row
row = { string(sheetLabel), string(keyA), string(keyB), ...
        numel(tt), bestName, ...
        rmE, rmL, rmG, ...
        rE,  rL,  rG, ...
        KL,  KG,  tLG, ...
        aE,  aL,  aG, ...
        dA(1), dA(2), dA(3), ...
        R2E, R2L, R2G };

% append t_to_pct fields at the end (caller converts to table + names)
row = [row, num2cell(t_to_pct)];
end

%% ---------- ANOVA: Do strains differ? ----------
%  (a) One-way ANOVA of Strain within each Condition
%  (b) Two-way ANOVA Strain × Condition across all replicates
metrics = { 'r_Exp_1_per_h', 'r_Log_1_per_h', 'r_Gomp_1_per_h', 'MaxDeriv_norm_1_per_h' };
run_anova_blocks(OutAll, metrics, ANOVA_XLSX, ANOVA_FIG_FOLDER, RUN_TAG);

fprintf('Saved ANOVA results to %s and figures to %s\n', ANOVA_XLSX, ANOVA_FIG_FOLDER);

end % === main ===



%% =================== HELPERS ===================

function [headerRow, detected] = detect_header_row(filepath, sheetLabel)
detected = false; headerRow = 1;
try
    rawPreview = readcell(filepath, 'Sheet', sheetLabel, 'Range', 'A1:ZZ100');
    for r = 1:size(rawPreview,1)
        rowVals = string(rawPreview(r,:));
        rowVals = lower(strtrim(rowVals));
        if any(contains(rowVals, ["time","time (h)","t (","elapsed","hour","hr","hrs","min","minute","minutes"]))
            headerRow = r; detected = true; return;
        end
    end
catch
    detected = false; headerRow = 1;
end
end

function [t_hours, valid_mask] = parse_time_column_improved(col)
n = numel(col);
t_hours = nan(n,1);

if isduration(col)
    t_hours = hours(col); [t_hours, valid_mask] = normalize_elapsed(t_hours); return;
end
if isdatetime(col)
    col = col(:); t0 = col(find(~isnat(col),1,'first'));
    t_hours = hours(col - t0); valid_mask = ~isnan(t_hours) & ~isnat(col); return;
end
if isnumeric(col)
    v = double(col);
    if all(isfinite(v))
        if max(v) <= 1.1 && any(v < 1), t_hours = v * 24; else, t_hours = v; end
    end
    [t_hours, valid_mask] = normalize_elapsed(t_hours); return;
end
if iscell(col)
    for i = 1:n, t_hours(i) = parse_time_any(col{i}); end
    [t_hours, valid_mask] = normalize_elapsed(t_hours); return;
end
if isstring(col) || ischar(col)
    s = string(col);
    for i = 1:n, t_hours(i) = parse_time_any(s(i)); end
    [t_hours, valid_mask] = normalize_elapsed(t_hours); return;
end
valid_mask = false(n,1);
end

function [th_norm, valid_mask] = normalize_elapsed(th)
th_norm = th;
valid_mask = isfinite(th_norm);
firstFinite = find(valid_mask,1,'first');
if ~isempty(firstFinite)
    t0 = th_norm(firstFinite);
    th_norm = th_norm - t0;
end
end

function th = parse_time_any(x)
th = NaN;
if isempty(x), return; end
if isnumeric(x)
    if isfinite(x)
        if x <= 1.1 && x >= 0, th = x*24; else, th = x; end
    end
    return;
end
s = string(x);
s = replace(s, char(160), ' ');
s = replace(s, char(8239), ' ');
s = lower(strtrim(regexprep(s, '\s+', ' ')));
s = replace(s, ',', '.');

tok = regexp(s, '^.*?(\d+)\s*:\s*(\d{1,2})(?::\s*(\d{1,2}))?.*$', 'tokens', 'once');
if ~isempty(tok)
    H = str2double(tok{1}); M = str2double(tok{2});
    S = 0; if numel(tok)>=3 && ~isempty(tok{3}), S = str2double(tok{3}); end
    th = H + M/60 + S/3600; return;
end
H = 0; M = 0; S = 0;
tokH = regexp(s, '(\d+(?:\.\d+)?)\s*(h|hr|hrs|hour|hours)', 'tokens', 'once'); if ~isempty(tokH), H = str2double(tokH{1}); end
tokM = regexp(s, '(\d+(?:\.\d+)?)\s*(m|min|mins|minute|minutes)', 'tokens', 'once'); if ~isempty(tokM), M = str2double(tokM{1}); end
tokS = regexp(s, '(\d+(?:\.\d+)?)\s*(s|sec|secs|second|seconds)', 'tokens', 'once'); if ~isempty(tokS), S = str2double(tokS{1}); end
if H>0 || M>0 || S>0, th = H + M/60 + S/3600; return; end

tok = regexp(s, '^.*?(\d+(?:\.\d+)?)\s*(m|min|mins|minute|minutes)\b.*$', 'tokens', 'once');
if ~isempty(tok), th = str2double(tok{1})/60; return; end

tok = regexp(s, '^.*?(\d+(?:\.\d+)?).*$','tokens','once');
if ~isempty(tok), th = str2double(tok{1}); return; end
end

function f = safeFilename(s)
f = string(s);
f = regexprep(f, '[:\\/<>|"?*]+', '_');
f = strtrim(f);
if strlength(f)==0, f = "replicate"; end
end

function A = localAICc(RSS, k, n)
if ~isfinite(RSS) || RSS <= 0 || n <= k + 1
    A = NaN; return;
end
A = n*log(RSS/n) + 2*k + (2*k*(k+1))/(n - k - 1);
end

function name = sanitizeToken(s)
s = strrep(s,'/','_per_');
s = regexprep(s,'[^A-Za-z0-9_]','_');
s = regexprep(s,'_+','_');
s = regexprep(s,'^_+|_+$','');
if isempty(s), s = 'units'; end
name = s;
end

function t = timeToPctK_Logistic(p, r, K, N0, EPS)
if ~isfinite(p) || p<=0 || p>=1 || ~all(isfinite([r,K,N0])) || r<=0 || K<=N0
    if isfinite(K) && isfinite(N0) && N0 >= p*K, t = 0; else, t = NaN; end
    return;
end
if N0 >= p*K, t = 0; return; end
A = (K - N0) / max(N0, EPS);
if A <= 0, t = NaN; return; end
rhs = (1/p - 1) / A;
t = -log(rhs)/r;
if ~isfinite(t) || t < 0, t = max(0,t); end
end

function t = timeToPctK_Gompertz(p, r, K, N0, tL_rel, EPS)
% tL_rel is Gompertz lag *relative to t0*
if ~isfinite(p) || p<=0 || p>=1 || ~all(isfinite([r,K,N0,tL_rel])) || r<=0 || K<=N0
    if isfinite(K) && isfinite(N0) && N0 >= p*K, t = 0; else, t = NaN; end
    return;
end
if N0 >= p*K, t = 0; return; end
z = K / max(N0, EPS);
if z <= 1, t = NaN; return; end
val = log(log(z)) - log(-log(p));
t = tL_rel + val / r;
if ~isfinite(t) || t < 0, t = max(0,t); end
end

%% ------- TITLE SPACING HELPER -------
function bump_title(ax, dy)
% Push the axes title up by dy (normalized). Use dy=0.02 per your spec.
if nargin<1 || isempty(ax), ax = gca; end
if nargin<2, dy = 0.02; end
t = ax.Title;
set(t,'Units','normalized');
pos = t.Position;
pos(2) = pos(2) + dy;
t.Position = pos;
end

%% ------- PLACEHOLDER FIG HELPER -------
function write_placeholder_fig(titleStr, noteStr, savePath)
try
    outDir = fileparts(savePath);
    if ~exist(outDir,'dir'), mkdir(outDir); end
    f = figure('Visible','off'); ax = axes(f); %#ok<LAXES>
    axis(ax,'off');
    text(0.5,0.62, titleStr, 'HorizontalAlignment','center', 'FontWeight','bold','Interpreter','none');
    text(0.5,0.40, noteStr,  'HorizontalAlignment','center', 'Interpreter','none');
    saveas(f, savePath);
    close(f);
catch ME
    warning('Failed to write placeholder to %s: %s', savePath);
end
end

%% ------- SUMMARY BOXPLOTS (RMSE, AICc, R²) -------
function make_summary_boxplots(RMSE_Exp, RMSE_Log, RMSE_Gom, ...
                               AIC_Exp,  AIC_Log,  AIC_Gom,  ...
                               R2_Exp,   R2_Log,   R2_Gom, ...
                               outFolder, RUN_TAG)
if ~exist(outFolder,'dir'), mkdir(outFolder); end

    function boxplot_grouped(dataCell, labels, ylab)
        vals = []; grp = [];
        for i=1:numel(dataCell)
            v = dataCell{i}; v = v(isfinite(v));
            vals = [vals; v]; %#ok<AGROW>
            grp  = [grp;  i*ones(numel(v),1)]; %#ok<AGROW>
        end
        if isempty(vals)
            write_placeholder_fig(sprintf('%s boxplot', ylab), 'No finite data available.', ...
                fullfile(outFolder, sprintf('BOX_%s__%s.png', regexprep(ylab,'\^|\\',''), RUN_TAG)));
            return;
        end
        boxplot(vals, grp, 'Labels', labels, 'LabelOrientation','inline');
        ylabel(ylab, 'Interpreter','tex');
    end

    function add_mean_sd_overlay(dataCell)
        hold on;
        for i=1:numel(dataCell)
            v = dataCell{i}; v = v(isfinite(v));
            if isempty(v), continue; end
            mu = mean(v); sd = std(v);
            errorbar(i, mu, sd, 'o', 'MarkerFaceColor','auto', 'LineWidth',1.2);
        end
    end

% RMSE
try
    data = {RMSE_Exp(:), RMSE_Log(:), RMSE_Gom(:)};
    labels = {'Exp','Logistic','Gompertz'};
    fig = figure('Visible','off');
    boxplot_grouped(data, labels, 'RMSE');
    add_mean_sd_overlay(data);
    title('RMSE across replicates'); bump_title(gca, 0.02); grid on;
    saveas(fig, fullfile(outFolder, sprintf('BOX_RMSE__%s.png', RUN_TAG))); close(fig);
catch ME, warning('RMSE boxplot failed: %s'); end

% AICc
try
    data = {AIC_Exp(:), AIC_Log(:), AIC_Gom(:)};
    labels = {'Exp','Logistic','Gompertz'};
    fig = figure('Visible','off');
    boxplot_grouped(data, labels, 'AICc');
    add_mean_sd_overlay(data);
    title('AICc across replicates'); bump_title(gca, 0.02); grid on;
    saveas(fig, fullfile(outFolder, sprintf('BOX_AICc__%s.png', RUN_TAG))); close(fig);
catch ME, warning('AICc boxplot failed: %s'); end

% R² (keep values as-is, clip texts)
try
    data = {R2_Exp(:), R2_Log(:), R2_Gom(:)};
    labels = {'Exp','Logistic','Gompertz'};
    fig = figure('Visible','off');
    boxplot_grouped(data, labels, 'R^2');
    add_mean_sd_overlay(data);
    ax = gca;
    lo = cellfun(@(v)min(v(isfinite(v))), data, 'UniformOutput', false);
    lo = [lo{:}]; lo = lo(isfinite(lo));
    ylo = 0; if ~isempty(lo), ylo = min([0, lo]); end
    ylim(ax, [ylo 1]);
    % annotate mean ± 95% CI (clipped)
    for i = 1:numel(data)
        v = data{i}; v = v(isfinite(v));
        if numel(v) >= 1
            mu = mean(v);
            ci = 0;
            if numel(v) >= 2
                ci = tinv(0.975, numel(v)-1)*std(v)/sqrt(numel(v));
            end
            ytxt = min(max(ax.YLim(1)+0.02, mu+ci+0.02), ax.YLim(2)-0.02);
            text(i+0.05, ytxt, sprintf('\\mu=%.3f, 95%% CI=\\pm%.3f', mu, ci), ...
                 'HorizontalAlignment','left','VerticalAlignment','bottom','Clipping','on');
        end
    end
    title('R^2 across replicates (with 95% CI of the mean)'); bump_title(gca, 0.02); grid on;
    saveas(fig, fullfile(outFolder, sprintf('BOX_R2__%s.png', RUN_TAG))); close(fig);
catch ME, warning('R^2 boxplot failed: %s'); end
end

%% ------- ANOVA HELPERS (ROBUST, ALWAYS OUTPUTS FIGS) -------
function run_anova_blocks(OutAll, metrics, anova_xlsx, fig_folder, RUN_TAG)
% Ensure categories
OutAll.Strain    = categorical(OutAll.Strain);
OutAll.Condition = categorical(OutAll.Condition);

anova_summary_rows = [];

for m = 1:numel(metrics)
    met = metrics{m};
    if ~ismember(met, OutAll.Properties.VariableNames)
        warning('ANOVA metric "%s" not found in table. Skipping.', met);
        continue;
    end

    % -------- One-way ANOVA per Condition (Strain differences within each Condition)
    conds = categories(OutAll.Condition);
    perCondRows = [];
    for ci = 1:numel(conds)
        cond = conds{ci};
        mask = OutAll.Condition == cond & isfinite(OutAll.(met)) ...
               & ~ismissing(OutAll.Strain) & ~ismissing(OutAll.Condition);

        y   = OutAll.(met)(mask);
        grp = OutAll.Strain(mask);

        if numel(y) < 3 || numel(categories(grp)) < 2
            write_placeholder_fig(sprintf('%s | Condition: %s', met, string(cond)), ...
                                  'Not enough data for one-way ANOVA.', ...
                                  fullfile(fig_folder, sprintf('ANOVA_%s__Cond_%s__%s.png', met, safeFilename(string(cond)), RUN_TAG)));
            continue;
        end

        try
            [p, tbl] = anova1(y, grp, 'off'); %#ok<ASGLU>
            [Fval, df1, df2] = parse_anova_tbl(tbl);
            perCondRows = [perCondRows; {met, cond, p, Fval, df1, df2, numel(y)}]; %#ok<AGROW>

            f = figure('Visible','off');
            boxplot(y, grp, 'LabelOrientation','inline');
            ylabel(met, 'Interpreter','none');
            title(sprintf('%s | Condition: %s', met, string(cond)), 'Interpreter','none');
            bump_title(gca, 0.02);
            grid on;
            saveas(f, fullfile(fig_folder, sprintf('ANOVA_%s__Cond_%s__%s.png', met, safeFilename(string(cond)), RUN_TAG)));
            close(f);

        catch E
            warning('anova1 failed for %s (Condition=%s): %s', met, string(cond));
            write_placeholder_fig(sprintf('%s | Condition: %s', met, string(cond)), ...
                                  'anova1 failed.', ...
                                  fullfile(fig_folder, sprintf('ANOVA_%s__Cond_%s__%s.png', met, safeFilename(string(cond)), RUN_TAG)));
        end
    end

    if ~isempty(perCondRows)
        T1 = cell2table(perCondRows, 'VariableNames', {'Metric','Condition','p_value','F','df1','df2','N'});
        writetable(T1, anova_xlsx, 'Sheet', sprintf('OneWay_%s', met));
        anova_summary_rows = [anova_summary_rows; [repmat({'OneWay'},height(T1),1), table2cell(T1)] ]; %#ok<AGROW>
    end

     % -------- Two-way ANOVA across all data: Strain, Condition, Interaction
    mask2 = isfinite(OutAll.(met)) & ~ismissing(OutAll.Strain) & ~ismissing(OutAll.Condition);

    y2   = OutAll.(met)(mask2);    y2   = y2(:);
    fac1 = OutAll.Strain(mask2);   fac1 = fac1(:);   % Strain (categorical)
    fac2 = OutAll.Condition(mask2);fac2 = fac2(:);   % Condition (categorical)

    if iscategorical(fac1), fac1 = removecats(fac1); end
    if iscategorical(fac2), fac2 = removecats(fac2); end

    okLength = numel(y2)==numel(fac1) && numel(y2)==numel(fac2);

    if ~okLength
        warning('Two-way ANOVA alignment failed for %s: lengths y=%d, fac1=%d, fac2=%d. Writing placeholder.', ...
                met, numel(y2), numel(fac1), numel(fac2));
        write_placeholder_fig(sprintf('Two-way ANOVA overview: %s (by Strain)', met), ...
                              'Length mismatch after alignment. Check missing data / masks.', ...
                              fullfile(fig_folder, sprintf('ANOVA_%s__TwoWay_byStrain__%s.png', met, RUN_TAG)));
    else
        % Need at least 2 levels in Strain, ≥1 in Condition, and ≥3 obs total
        if numel(y2) >= 3 && numel(categories(fac1)) >= 2 && numel(categories(fac2)) >= 1

        % --- primary path: ANOVAN with integer groups
        usedFallback = false;
        try
            [g1, ~] = grp2idx(fac1);
            [g2, ~] = grp2idx(fac2);
            p2 = anovan(double(y2), {g1, g2}, ...
                        'model','interaction', ...
                        'varnames', {'Strain','Condition'}, ...
                        'display','off');
        
            tw_names = {'Strain','Condition','Strain:Condition'};
            k = min(3, numel(p2));
            TW = table(tw_names(1:k)', p2(1:k)', 'VariableNames', {'Effect','p_value'});
            writetable(TW, anova_xlsx, 'Sheet', sprintf('TwoWay_%s', met));
        
        catch E  %#ok<NASGU>
            % --- fallback: linear model ANOVA
            warning('anovan failed for %s. Falling back to fitlm.', met);
            usedFallback = true;
            try
                Tfit = table(double(y2), categorical(fac1), categorical(fac2), ...
                             'VariableNames', {'Y','Strain','Condition'});
                mdl = fitlm(Tfit, 'Y ~ Strain*Condition');   % includes interaction
                A = anova(mdl,'summary');                    % table with p-values
        
                % term labels can be in A.Source (newer) or RowNames (older)
                if ismember('Source', A.Properties.VariableNames)
                    rowLabels = string(A.Source);
                else
                    rowLabels = string(A.Properties.RowNames);
                end
                rowLabels = strtrim(rowLabels);
        
                pS  = NaN; pC = NaN; pSC = NaN;
                iS  = find(rowLabels == "Strain", 1);
                iC  = find(rowLabels == "Condition", 1);
                iSC = find(rowLabels == "Strain:Condition", 1);
                if ~isempty(iS),  pS  = A.pValue(iS);  end
                if ~isempty(iC),  pC  = A.pValue(iC);  end
                if ~isempty(iSC), pSC = A.pValue(iSC); end
        
                TW = table({'Strain';'Condition';'Strain:Condition'}, [pS;pC;pSC], ...
                           'VariableNames', {'Effect','p_value'});
                writetable(TW, anova_xlsx, 'Sheet', sprintf('TwoWay_%s', met));
        
            catch E2
                warning('fitlm fallback also failed for %s: %s. Writing placeholder sheet.', met, E2.message);
                TW = table({'Strain';'Condition';'Strain:Condition'}, [NaN;NaN;NaN], ...
                           'VariableNames', {'Effect','p_value'});
                writetable(TW, anova_xlsx, 'Sheet', sprintf('TwoWay_%s', met));
            end
        end

            % --- Overview figure (always create one)
            fig2 = figure('Visible','off'); hold on;
            try
                if exist('boxchart','file') == 2
                    boxchart(fac1, y2);
                else
                    [~,~,gidx] = unique(fac1);
                    boxplot(y2, gidx, 'Labels', cellstr(categories(removecats(fac1))), 'LabelOrientation','inline');
                end
                ylabel(met, 'Interpreter','none');
                ttl = sprintf('Two-way ANOVA overview: %s (by Strain)%s', ...
                               met, usedFallback*' [fitlm]');
                title(ttl, 'Interpreter','none');
                bump_title(gca, 0.02);
                grid on;
                saveas(fig2, fullfile(fig_folder, sprintf('ANOVA_%s__TwoWay_byStrain__%s.png', met, RUN_TAG)));
            catch E3
                warning('Two-way overview plot failed for %s: %s. Writing placeholder.', met, E3.message);
                try, close(fig2); end %#ok<TRYNC>
                write_placeholder_fig(sprintf('Two-way ANOVA overview: %s (by Strain)', met), ...
                                      'Plotting error (boxchart/boxplot).', ...
                                      fullfile(fig_folder, sprintf('ANOVA_%s__TwoWay_byStrain__%s.png', met, RUN_TAG)));
            end
            close(fig2);

        else
            write_placeholder_fig(sprintf('Two-way ANOVA overview: %s (by Strain)', met), ...
                                  'Not enough data for two-way ANOVA (need ≥3 points, ≥2 strains).', ...
                                  fullfile(fig_folder, sprintf('ANOVA_%s__TwoWay_byStrain__%s.png', met, RUN_TAG)));
        end
    end
end

% Optional: compact Summary sheet
if ~isempty(anova_summary_rows)
    Tsum = cell2table(anova_summary_rows, 'VariableNames', {'Test','Metric','Condition','p_value','F','df1','df2','N'});
    writetable(Tsum, anova_xlsx, 'Sheet', 'Summary');
end
end

function [Fval, df1, df2] = parse_anova_tbl(tblCell)
% tblCell from anova1 is a cell array with header; rows include {'Groups', SS, df, MS, F, Prob>F}
Fval = NaN; df1 = NaN; df2 = NaN;
try
    hdr = tblCell(1,:);
    rowNames = tblCell(2:end,1);
    colF = find(strcmp(hdr,'F'));
    colDf = find(strcmp(hdr,'df'));
    rGroups = find(strcmp(rowNames,'Groups')) + 1; % +1 for header row
    rError  = find(strcmp(rowNames,'Error'))  + 1;

    if ~isempty(rGroups) && ~isempty(colF)
        Fval = asnum(tblCell{rGroups, colF});
    end
    if ~isempty(rGroups) && ~isempty(rError) && ~isempty(colDf)
        df1 = asnum(tblCell{rGroups, colDf});
        df2 = asnum(tblCell{rError,  colDf});
    end
catch
    % leave NaNs
end
end

function v = asnum(x)
if isnumeric(x), v = x; return; end
if isstring(x) || ischar(x), v = str2double(string(x)); return; end
v = NaN;
end

%% ------- 96-WELL PLATE LAYOUT HELPERS -------
function S = read_plate_layout(platePath)
% Fixed mapping:
%   Row 1  = Bacterial strains
%   Col 1  = Initial concentrations
raw = readcell(platePath, 'Sheet', 1);
if size(raw,1) < 2 || size(raw,2) < 2
    error('Plate layout: expected at least a 2x2 grid.');
end
axisTop  = "Strain";
axisLeft = "Initial Concentration";
M = containers.Map('KeyType','char','ValueType','any');
for r = 2:size(raw,1)
    for c = 2:size(raw,2)
        nm = raw{r,c};
        if ~(ischar(nm) || isstring(nm)), continue; end
        sName = strtrim(string(nm)); if strlength(sName) == 0, continue; end
        strainVal   = strtrim(string(raw{1,c}));
        initConcVal = strtrim(string(raw{r,1}));
        key = normKey(sName);
        M(char(key)) = struct('Condition',string(initConcVal),'Strain',string(strainVal));
    end
end
S = struct('map', M, 'axisTop', axisTop, 'axisLeft', axisLeft);
end

function k = normKey(s)
k = lower(strtrim(string(s)));
end